# Design Document - Modul Pendaftaran Peserta Didik

## Overview

Modul Pendaftaran Peserta Didik adalah sistem manajemen pendaftaran siswa baru yang terpisah dari modul PPDB yang sudah ada. Modul ini akan dibangun dalam folder terpisah dengan struktur yang modular dan menggunakan role-based access control untuk memisahkan hak akses Admin, Operator, dan Siswa.

**Perbedaan dengan Modul PPDB yang Ada:**
- Folder struktur yang benar-benar terpisah (`pendaftaran/`)
- Sistem autentikasi dan session management yang independen
- Database schema yang lebih lengkap dengan audit trail
- UI/UX yang lebih modern dengan responsive design
- Fitur export laporan (Excel/PDF)
- Dashboard analytics yang lebih detail

## Architecture

### Folder Structure

```
ppdb/
├── config/
│   ├── database.php          # Konfigurasi koneksi database
│   └── constants.php          # Konstanta aplikasi
├── includes/
│   ├── auth.php              # Fungsi autentikasi
│   ├── functions.php         # Helper functions
│   ├── security.php          # Security functions
│   └── db.php                # Database connection
├── assets/
│   ├── css/
│   │   ├── admin.css         # Style untuk admin/operator
│   │   └── public.css        # Style untuk public/siswa
│   ├── js/
│   │   ├── admin.js          # JavaScript admin
│   │   ├── form-validation.js # Validasi form
│   │   └── chart.js          # Library untuk grafik
│   └── images/
│       └── logo.png
├── uploads/
│   └── documents/            # Folder upload dokumen
│       └── .htaccess         # Proteksi akses langsung
├── public/
│   ├── index.php             # Landing page pendaftaran
│   ├── register.php          # Form pendaftaran siswa
│   ├── check-status.php      # Cek status pendaftaran
│   ├── print-card.php        # Cetak kartu pendaftaran
│   └── login.php             # Login untuk siswa
├── admin/
│   ├── index.php             # Dashboard admin
│   ├── registrations.php     # Daftar pendaftaran
│   ├── registration-detail.php # Detail pendaftaran
│   ├── settings.php          # Pengaturan sistem
│   ├── reports.php           # Laporan dan statistik
│   ├── users.php             # Manajemen user
│   └── logs.php              # Activity logs
├── operator/
│   ├── index.php             # Dashboard operator
│   ├── registrations.php     # Daftar pendaftaran
│   └── registration-detail.php # Detail & verifikasi
├── api/
│   ├── registration.php      # API untuk pendaftaran
│   ├── status.php            # API cek status
│   └── export.php            # API export data
├── database/
│   └── schema.sql            # Database schema
├── .htaccess                 # URL rewriting & security
├── README.md                 # Dokumentasi
└── install.php               # Installer script
```

### Technology Stack

- **Backend**: PHP 7.4+
- **Database**: MySQL 5.7+ / MariaDB 10.3+
- **Frontend**: HTML5, CSS3, JavaScript (Vanilla)
- **Libraries**: 
  - Chart.js untuk visualisasi data
  - PHPSpreadsheet untuk export Excel
  - TCPDF untuk export PDF
  - Bootstrap 5 untuk responsive UI

## Components and Interfaces

### 1. Authentication System

**File**: `includes/auth.php`

```php
class Auth {
  // Login user dengan role validation
  public static function login($username, $password, $role)
  
  // Logout dan destroy session
  public static function logout()
  
  // Check apakah user sudah login
  public static function isLoggedIn()
  
  // Check role user
  public static function hasRole($role)
  
  // Get current user data
  public static function getCurrentUser()
  
  // Redirect jika tidak punya akses
  public static function requireRole($role)
}
```

**Session Structure:**
```php
$_SESSION['user'] = [
  'id' => 1,
  'username' => 'admin',
  'full_name' => 'Administrator',
  'role' => 'admin', // admin, operator, siswa
  'login_time' => '2024-01-01 10:00:00'
];
```

### 2. Database Layer

**File**: `includes/db.php`

```php
class Database {
  private static $instance = null;
  private $connection;
  
  // Singleton pattern untuk koneksi
  public static function getInstance()
  
  // Execute query dengan prepared statement
  public function query($sql, $params = [])
  
  // Insert data dan return last insert ID
  public function insert($table, $data)
  
  // Update data
  public function update($table, $data, $where)
  
  // Delete data
  public function delete($table, $where)
  
  // Get single row
  public function getRow($sql, $params = [])
  
  // Get multiple rows
  public function getRows($sql, $params = [])
}
```

### 3. Registration Handler

**File**: `api/registration.php`

```php
class RegistrationHandler {
  // Validate form data
  public function validateData($data)
  
  // Generate unique registration number
  public function generateRegistrationNumber()
  
  // Handle file upload
  public function uploadDocument($file, $type)
  
  // Save registration to database
  public function saveRegistration($data)
  
  // Send confirmation email/SMS
  public function sendConfirmation($registration_id)
}
```

### 4. Status Manager

**File**: `includes/functions.php`

```php
class StatusManager {
  // Get registration by number
  public function getRegistration($reg_number)
  
  // Update status
  public function updateStatus($id, $status, $notes = '')
  
  // Get status history
  public function getStatusHistory($id)
  
  // Check if can change status
  public function canChangeStatus($current_status, $new_status, $role)
}
```

### 5. Report Generator

**File**: `api/export.php`

```php
class ReportGenerator {
  // Export to Excel
  public function exportToExcel($data, $filename)
  
  // Export to PDF
  public function exportToPDF($data, $filename)
  
  // Generate statistics
  public function generateStatistics($filters = [])
  
  // Generate chart data
  public function getChartData($type, $filters = [])
}
```

## Data Models

### Database Schema

#### 1. Table: `registration_users`
Menyimpan data user (admin, operator, siswa)

```sql
CREATE TABLE registration_users (
  id INT AUTO_INCREMENT PRIMARY KEY,
  username VARCHAR(50) UNIQUE NOT NULL,
  password VARCHAR(255) NOT NULL,
  full_name VARCHAR(255) NOT NULL,
  email VARCHAR(255),
  phone VARCHAR(20),
  role ENUM('admin', 'operator', 'siswa') NOT NULL,
  is_active TINYINT(1) DEFAULT 1,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  last_login TIMESTAMP NULL,
  INDEX idx_username (username),
  INDEX idx_role (role)
);
```

#### 2. Table: `student_registrations`
Menyimpan data pendaftaran siswa

```sql
CREATE TABLE student_registrations (
  id INT AUTO_INCREMENT PRIMARY KEY,
  registration_number VARCHAR(50) UNIQUE NOT NULL,
  user_id INT NULL, -- Link ke registration_users jika siswa punya akun
  
  -- Data Pribadi
  full_name VARCHAR(255) NOT NULL,
  nickname VARCHAR(100),
  gender ENUM('L', 'P') NOT NULL,
  birth_place VARCHAR(100) NOT NULL,
  birth_date DATE NOT NULL,
  religion VARCHAR(50) NOT NULL,
  nik VARCHAR(20),
  nisn VARCHAR(20),
  
  -- Alamat
  address TEXT NOT NULL,
  rt VARCHAR(5),
  rw VARCHAR(5),
  kelurahan VARCHAR(100),
  kecamatan VARCHAR(100),
  city VARCHAR(100) NOT NULL,
  province VARCHAR(100) NOT NULL,
  postal_code VARCHAR(10),
  
  -- Kontak
  phone VARCHAR(20),
  email VARCHAR(255),
  
  -- Data Orang Tua
  father_name VARCHAR(255),
  father_nik VARCHAR(20),
  father_occupation VARCHAR(100),
  father_phone VARCHAR(20),
  father_education VARCHAR(50),
  father_income VARCHAR(50),
  
  mother_name VARCHAR(255),
  mother_nik VARCHAR(20),
  mother_occupation VARCHAR(100),
  mother_phone VARCHAR(20),
  mother_education VARCHAR(50),
  mother_income VARCHAR(50),
  
  -- Data Wali (jika ada)
  guardian_name VARCHAR(255),
  guardian_nik VARCHAR(20),
  guardian_relation VARCHAR(50),
  guardian_phone VARCHAR(20),
  guardian_address TEXT,
  
  -- Data Sekolah Asal
  previous_school VARCHAR(255) NOT NULL,
  previous_school_npsn VARCHAR(20),
  previous_school_address TEXT,
  graduation_year YEAR NOT NULL,
  exam_number VARCHAR(50),
  
  -- Pilihan Jurusan
  major_choice_1 VARCHAR(100) NOT NULL,
  major_choice_2 VARCHAR(100),
  
  -- Prestasi (JSON)
  achievements TEXT, -- JSON array
  
  -- Status & Workflow
  status ENUM('pending', 'verified', 'accepted', 'rejected') DEFAULT 'pending',
  status_notes TEXT,
  verified_by INT NULL,
  verified_at TIMESTAMP NULL,
  accepted_by INT NULL,
  accepted_at TIMESTAMP NULL,
  
  -- Metadata
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  
  FOREIGN KEY (user_id) REFERENCES registration_users(id) ON DELETE SET NULL,
  FOREIGN KEY (verified_by) REFERENCES registration_users(id) ON DELETE SET NULL,
  FOREIGN KEY (accepted_by) REFERENCES registration_users(id) ON DELETE SET NULL,
  INDEX idx_registration_number (registration_number),
  INDEX idx_status (status),
  INDEX idx_created_at (created_at)
);
```

#### 3. Table: `registration_documents`
Menyimpan dokumen upload

```sql
CREATE TABLE registration_documents (
  id INT AUTO_INCREMENT PRIMARY KEY,
  registration_id INT NOT NULL,
  document_type ENUM('photo', 'birth_certificate', 'family_card', 'report_card', 'certificate', 'other') NOT NULL,
  file_name VARCHAR(255) NOT NULL,
  file_path VARCHAR(500) NOT NULL,
  file_size INT NOT NULL,
  mime_type VARCHAR(100),
  uploaded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  
  FOREIGN KEY (registration_id) REFERENCES student_registrations(id) ON DELETE CASCADE,
  INDEX idx_registration_id (registration_id)
);
```

#### 4. Table: `registration_settings`
Pengaturan sistem

```sql
CREATE TABLE registration_settings (
  id INT AUTO_INCREMENT PRIMARY KEY,
  setting_key VARCHAR(100) UNIQUE NOT NULL,
  setting_value TEXT,
  setting_type VARCHAR(50) DEFAULT 'string',
  description TEXT,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  updated_by INT NULL,
  
  FOREIGN KEY (updated_by) REFERENCES registration_users(id) ON DELETE SET NULL
);

-- Default settings
INSERT INTO registration_settings (setting_key, setting_value, setting_type, description) VALUES
('registration_open', '1', 'boolean', 'Status buka/tutup pendaftaran'),
('registration_start_date', '2024-01-01', 'date', 'Tanggal mulai pendaftaran'),
('registration_end_date', '2024-12-31', 'date', 'Tanggal akhir pendaftaran'),
('academic_year', '2024/2025', 'string', 'Tahun ajaran'),
('max_quota', '500', 'integer', 'Kuota maksimal siswa'),
('allowed_file_types', 'jpg,jpeg,png,pdf', 'string', 'Tipe file yang diizinkan'),
('max_file_size', '2097152', 'integer', 'Ukuran maksimal file (bytes)'),
('school_name', 'SMK Negeri 1', 'string', 'Nama sekolah'),
('school_address', '', 'text', 'Alamat sekolah'),
('contact_email', 'info@sekolah.sch.id', 'string', 'Email kontak'),
('contact_phone', '021-12345678', 'string', 'Telepon kontak');
```

#### 5. Table: `registration_logs`
Activity logs untuk audit trail

```sql
CREATE TABLE registration_logs (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NULL,
  registration_id INT NULL,
  action VARCHAR(100) NOT NULL,
  description TEXT,
  old_value TEXT,
  new_value TEXT,
  ip_address VARCHAR(45),
  user_agent TEXT,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  
  FOREIGN KEY (user_id) REFERENCES registration_users(id) ON DELETE SET NULL,
  FOREIGN KEY (registration_id) REFERENCES student_registrations(id) ON DELETE CASCADE,
  INDEX idx_user_id (user_id),
  INDEX idx_registration_id (registration_id),
  INDEX idx_created_at (created_at)
);
```

#### 6. Table: `registration_majors`
Daftar jurusan yang tersedia

```sql
CREATE TABLE registration_majors (
  id INT AUTO_INCREMENT PRIMARY KEY,
  code VARCHAR(20) UNIQUE NOT NULL,
  name VARCHAR(255) NOT NULL,
  description TEXT,
  quota INT DEFAULT 0,
  is_active TINYINT(1) DEFAULT 1,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);
```

## Error Handling

### Error Response Structure

```php
// Success response
[
  'success' => true,
  'message' => 'Data berhasil disimpan',
  'data' => [...],
  'registration_number' => 'REG2024001'
]

// Error response
[
  'success' => false,
  'message' => 'Terjadi kesalahan',
  'errors' => [
    'field_name' => 'Error message'
  ],
  'code' => 'ERROR_CODE'
]
```

### Error Codes

- `AUTH_FAILED`: Autentikasi gagal
- `PERMISSION_DENIED`: Tidak punya hak akses
- `VALIDATION_ERROR`: Error validasi input
- `FILE_UPLOAD_ERROR`: Error upload file
- `DATABASE_ERROR`: Error database
- `NOT_FOUND`: Data tidak ditemukan
- `REGISTRATION_CLOSED`: Pendaftaran ditutup
- `QUOTA_FULL`: Kuota penuh

### Error Logging

Semua error akan dicatat ke:
1. Database (`registration_logs` table)
2. File log (`logs/error.log`)
3. PHP error log (untuk critical errors)

```php
function logError($error_code, $message, $context = []) {
  // Log to database
  $db = Database::getInstance();
  $db->insert('registration_logs', [
    'user_id' => $_SESSION['user']['id'] ?? null,
    'action' => 'ERROR',
    'description' => $message,
    'old_value' => json_encode($context),
    'ip_address' => $_SERVER['REMOTE_ADDR'],
    'user_agent' => $_SERVER['HTTP_USER_AGENT']
  ]);
  
  // Log to file
  error_log(date('Y-m-d H:i:s') . " [$error_code] $message\n", 3, 'logs/error.log');
}
```

## Testing Strategy

### 1. Unit Testing

**Tools**: PHPUnit

**Test Coverage:**
- Authentication functions
- Validation functions
- Database operations
- File upload handling
- Registration number generation

**Example Test:**
```php
class RegistrationTest extends TestCase {
  public function testGenerateRegistrationNumber() {
    $handler = new RegistrationHandler();
    $regNumber = $handler->generateRegistrationNumber();
    $this->assertMatchesRegularExpression('/^REG\d{7}$/', $regNumber);
  }
  
  public function testValidateData() {
    $handler = new RegistrationHandler();
    $data = ['full_name' => '', 'email' => 'invalid'];
    $result = $handler->validateData($data);
    $this->assertFalse($result['success']);
    $this->assertArrayHasKey('errors', $result);
  }
}
```

### 2. Integration Testing

**Test Scenarios:**
- Complete registration flow (form submission → database → confirmation)
- Status update workflow (pending → verified → accepted)
- File upload and retrieval
- Authentication and authorization
- Report generation

### 3. Security Testing

**Test Areas:**
- SQL Injection prevention
- XSS prevention
- CSRF protection
- File upload security
- Session hijacking prevention
- Role-based access control

**Security Checklist:**
- [ ] All inputs sanitized and validated
- [ ] Prepared statements for all queries
- [ ] CSRF tokens on all forms
- [ ] File type and size validation
- [ ] Secure session configuration
- [ ] Password hashing (bcrypt)
- [ ] HTTPS enforcement
- [ ] Rate limiting on login

### 4. User Acceptance Testing

**Test Cases:**

**Siswa:**
- [ ] Dapat mengakses form pendaftaran
- [ ] Dapat mengisi dan submit form
- [ ] Dapat upload dokumen
- [ ] Dapat cek status pendaftaran
- [ ] Dapat cetak kartu jika diterima

**Operator:**
- [ ] Dapat login ke sistem
- [ ] Dapat melihat daftar pendaftaran
- [ ] Dapat verifikasi dokumen
- [ ] Dapat update status
- [ ] Dapat mencari data

**Admin:**
- [ ] Semua fitur operator
- [ ] Dapat mengatur pengaturan sistem
- [ ] Dapat melihat laporan
- [ ] Dapat export data
- [ ] Dapat manage users

### 5. Performance Testing

**Metrics:**
- Page load time < 2 seconds
- Form submission < 3 seconds
- File upload < 5 seconds (2MB file)
- Report generation < 10 seconds (1000 records)
- Database query < 100ms

**Load Testing:**
- Concurrent users: 100
- Requests per second: 50
- Database connections: 20

## Security Considerations

### 1. Input Validation

```php
function sanitizeInput($data) {
  if (is_array($data)) {
    return array_map('sanitizeInput', $data);
  }
  return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}
```

### 2. File Upload Security

```php
function validateUpload($file) {
  $allowed_types = ['image/jpeg', 'image/png', 'application/pdf'];
  $max_size = 2 * 1024 * 1024; // 2MB
  
  if (!in_array($file['type'], $allowed_types)) {
    throw new Exception('Tipe file tidak diizinkan');
  }
  
  if ($file['size'] > $max_size) {
    throw new Exception('Ukuran file terlalu besar');
  }
  
  // Check actual file type (not just extension)
  $finfo = finfo_open(FILEINFO_MIME_TYPE);
  $mime = finfo_file($finfo, $file['tmp_name']);
  finfo_close($finfo);
  
  if (!in_array($mime, $allowed_types)) {
    throw new Exception('Tipe file tidak valid');
  }
  
  return true;
}
```

### 3. CSRF Protection

```php
function generateCSRFToken() {
  if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
  }
  return $_SESSION['csrf_token'];
}

function validateCSRFToken($token) {
  return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}
```

### 4. Rate Limiting

```php
function checkRateLimit($action, $limit = 5, $period = 60) {
  $key = $action . '_' . $_SERVER['REMOTE_ADDR'];
  $attempts = $_SESSION['rate_limit'][$key] ?? [];
  
  // Remove old attempts
  $attempts = array_filter($attempts, function($time) use ($period) {
    return (time() - $time) < $period;
  });
  
  if (count($attempts) >= $limit) {
    throw new Exception('Terlalu banyak percobaan. Coba lagi nanti.');
  }
  
  $attempts[] = time();
  $_SESSION['rate_limit'][$key] = $attempts;
  
  return true;
}
```

## UI/UX Design

### Design Principles

1. **Responsive**: Mobile-first design, works on all devices
2. **Accessible**: WCAG 2.1 Level AA compliance
3. **Intuitive**: Clear navigation and user flow
4. **Consistent**: Unified design language across all pages
5. **Fast**: Optimized loading and minimal HTTP requests

### Color Scheme

```css
:root {
  --primary-color: #2563eb;      /* Blue */
  --secondary-color: #64748b;    /* Slate */
  --success-color: #10b981;      /* Green */
  --warning-color: #f59e0b;      /* Amber */
  --danger-color: #ef4444;       /* Red */
  --info-color: #06b6d4;         /* Cyan */
  --light-bg: #f8fafc;
  --dark-text: #1e293b;
  --border-color: #e2e8f0;
}
```

### Typography

```css
body {
  font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
  font-size: 16px;
  line-height: 1.6;
  color: var(--dark-text);
}

h1 { font-size: 2rem; font-weight: 700; }
h2 { font-size: 1.5rem; font-weight: 600; }
h3 { font-size: 1.25rem; font-weight: 600; }
```

### Component Library

**Buttons:**
```html
<button class="btn btn-primary">Primary</button>
<button class="btn btn-secondary">Secondary</button>
<button class="btn btn-success">Success</button>
```

**Forms:**
```html
<div class="form-group">
  <label for="input">Label</label>
  <input type="text" id="input" class="form-control" required>
  <small class="form-text">Helper text</small>
</div>
```

**Cards:**
```html
<div class="card">
  <div class="card-header">Header</div>
  <div class="card-body">Content</div>
  <div class="card-footer">Footer</div>
</div>
```

**Status Badges:**
```html
<span class="badge badge-pending">Pending</span>
<span class="badge badge-verified">Verified</span>
<span class="badge badge-accepted">Accepted</span>
<span class="badge badge-rejected">Rejected</span>
```

## Deployment Considerations

### Server Requirements

- PHP 7.4 or higher
- MySQL 5.7+ or MariaDB 10.3+
- Apache 2.4+ with mod_rewrite
- PHP Extensions: PDO, GD, mbstring, zip
- Minimum 512MB RAM
- 1GB disk space

### Installation Steps

1. Upload files to server
2. Create database
3. Import schema.sql
4. Configure database connection in `config/database.php`
5. Set folder permissions (uploads/, logs/)
6. Run install.php
7. Create admin user
8. Configure settings

### Backup Strategy

- Daily database backup
- Weekly full backup (files + database)
- Retention: 30 days
- Automated via cron job

### Monitoring

- Error logs monitoring
- Database performance
- Disk space usage
- User activity tracking
- Uptime monitoring
