<?php
/**
 * Database Backup & Restore Module
 * Modul untuk backup dan restore database SQLite
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';

// Check if user is logged in as admin
require_login();

$pageTitle = 'Backup & Restore Database';
$success = '';
$error = '';

// Configuration
$backupDir = __DIR__ . '/../backups/database';
$dataDir = __DIR__ . '/../data';
$dbFile = $dataDir . '/school.db';

if (!is_dir($backupDir)) {
    mkdir($backupDir, 0755, true);
}

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    // Backup Database (SQLite format)
    if ($action === 'backup') {
        try {
            $filename = 'backup_' . date('Y-m-d_H-i-s') . '.db';
            $filepath = $backupDir . '/' . $filename;
            
            if (file_exists($dbFile)) {
                if (copy($dbFile, $filepath)) {
                    $success = "Backup berhasil dibuat: {$filename} (" . formatFileSize(filesize($filepath)) . ")";
                } else {
                    $error = "Gagal membuat backup!";
                }
            } else {
                $error = "File database tidak ditemukan!";
            }
        } catch (Exception $e) {
            $error = "Error: " . $e->getMessage();
        }
    }
    
    // Backup Database to SQL format
    elseif ($action === 'backup_sql') {
        try {
            $format = $_POST['sql_format'] ?? 'sqlite';
            $filename = 'backup_' . date('Y-m-d_H-i-s') . ($format === 'mysql' ? '_mysql' : '') . '.sql';
            $filepath = $backupDir . '/' . $filename;
            
            $result = exportDatabaseToSQL($filepath, $format);
            if ($result['success']) {
                $formatName = $format === 'mysql' ? 'MySQL' : 'SQLite';
                $success = "Backup SQL ({$formatName}) berhasil dibuat: {$filename} (" . formatFileSize(filesize($filepath)) . ")";
            } else {
                $error = "Gagal membuat backup SQL: " . $result['message'];
            }
        } catch (Exception $e) {
            $error = "Error: " . $e->getMessage();
        }
    }
    
    // Restore Database
    elseif ($action === 'restore') {
        $filename = $_POST['filename'] ?? '';
        $filepath = $backupDir . '/' . $filename;
        
        if (!file_exists($filepath)) {
            $error = "File backup tidak ditemukan!";
        } else {
            try {
                // Backup current database first
                $tempBackup = $dbFile . '.before_restore_' . time();
                copy($dbFile, $tempBackup);
                
                // Restore from backup
                if (copy($filepath, $dbFile)) {
                    $success = "Database berhasil di-restore dari: {$filename}";
                    // Remove temp backup after successful restore
                    if (file_exists($tempBackup)) {
                        unlink($tempBackup);
                    }
                } else {
                    $error = "Gagal restore database!";
                    // Restore from temp backup
                    if (file_exists($tempBackup)) {
                        copy($tempBackup, $dbFile);
                        unlink($tempBackup);
                    }
                }
            } catch (Exception $e) {
                $error = "Error: " . $e->getMessage();
            }
        }
    }
    
    // Delete Backup
    elseif ($action === 'delete') {
        $filename = $_POST['filename'] ?? '';
        $filepath = $backupDir . '/' . $filename;
        
        if (file_exists($filepath)) {
            if (unlink($filepath)) {
                $success = "Backup berhasil dihapus: {$filename}";
            } else {
                $error = "Gagal menghapus backup!";
            }
        } else {
            $error = "File tidak ditemukan!";
        }
    }
    
    // Download Backup
    elseif ($action === 'download') {
        $filename = $_POST['filename'] ?? '';
        $filepath = $backupDir . '/' . $filename;
        
        if (file_exists($filepath)) {
            header('Content-Type: application/octet-stream');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            header('Content-Length: ' . filesize($filepath));
            readfile($filepath);
            exit;
        }
    }
}

// Function: Export database to SQL format
function exportDatabaseToSQL($filepath, $format = 'sqlite') {
    try {
        $db = get_db();
        
        if ($format === 'mysql') {
            // MySQL/MariaDB compatible format
            $output = "-- MySQL Database Backup (Converted from SQLite)\n";
            $output .= "-- Generated: " . date('Y-m-d H:i:s') . "\n";
            $output .= "-- Source Database: school.db\n";
            $output .= "-- Compatible with: MySQL 5.7+, MariaDB 10.2+\n\n";
            $output .= "SET SQL_MODE = \"NO_AUTO_VALUE_ON_ZERO\";\n";
            $output .= "SET time_zone = \"+00:00\";\n\n";
            $output .= "/*!40101 SET @OLD_CHARACTER_SET_CLIENT=@@CHARACTER_SET_CLIENT */;\n";
            $output .= "/*!40101 SET @OLD_CHARACTER_SET_RESULTS=@@CHARACTER_SET_RESULTS */;\n";
            $output .= "/*!40101 SET @OLD_COLLATION_CONNECTION=@@COLLATION_CONNECTION */;\n";
            $output .= "/*!40101 SET NAMES utf8mb4 */;\n\n";
        } else {
            // SQLite format
            $output = "-- SQLite Database Backup\n";
            $output .= "-- Generated: " . date('Y-m-d H:i:s') . "\n";
            $output .= "-- Database: school.db\n\n";
        }
        
        // Get all tables
        $tables = $db->query("SELECT name FROM sqlite_master WHERE type='table' AND name NOT LIKE 'sqlite_%' ORDER BY name")->fetchAll(PDO::FETCH_COLUMN);
        
        foreach ($tables as $table) {
            $output .= "-- --------------------------------------------------------\n";
            $output .= "-- Table structure for table `{$table}`\n";
            $output .= "-- --------------------------------------------------------\n\n";
            $output .= "DROP TABLE IF EXISTS `{$table}`;\n";
            
            if ($format === 'mysql') {
                // Convert SQLite CREATE TABLE to MySQL format
                $createStmt = $db->query("SELECT sql FROM sqlite_master WHERE type='table' AND name='{$table}'")->fetchColumn();
                $mysqlCreate = convertSQLiteToMySQL($createStmt, $table);
                $output .= $mysqlCreate . ";\n\n";
            } else {
                // Original SQLite format
                $createStmt = $db->query("SELECT sql FROM sqlite_master WHERE type='table' AND name='{$table}'")->fetchColumn();
                $output .= $createStmt . ";\n\n";
            }
            
            // Get data
            $rows = $db->query("SELECT * FROM `{$table}`")->fetchAll(PDO::FETCH_ASSOC);
            
            if (!empty($rows)) {
                $output .= "-- Dumping data for table `{$table}`\n\n";
                
                if ($format === 'mysql') {
                    $output .= "INSERT INTO `{$table}` ";
                    
                    // Get column names
                    $columns = array_keys($rows[0]);
                    $output .= "(`" . implode('`, `', $columns) . "`) VALUES\n";
                    
                    $rowCount = count($rows);
                    foreach ($rows as $index => $row) {
                        $values = [];
                        foreach ($row as $value) {
                            if ($value === null) {
                                $values[] = 'NULL';
                            } elseif (is_numeric($value)) {
                                $values[] = $value;
                            } else {
                                // Escape for MySQL
                                $escaped = str_replace(['\\', "'", '"', "\n", "\r", "\x00", "\x1a"], 
                                                      ['\\\\', "\\'", '\\"', '\\n', '\\r', '\\0', '\\Z'], 
                                                      $value);
                                $values[] = "'" . $escaped . "'";
                            }
                        }
                        $output .= "(" . implode(', ', $values) . ")";
                        $output .= ($index < $rowCount - 1) ? ",\n" : ";\n";
                    }
                } else {
                    // SQLite format
                    foreach ($rows as $row) {
                        $values = [];
                        foreach ($row as $value) {
                            if ($value === null) {
                                $values[] = 'NULL';
                            } elseif (is_numeric($value)) {
                                $values[] = $value;
                            } else {
                                $values[] = "'" . str_replace("'", "''", $value) . "'";
                            }
                        }
                        $output .= "INSERT INTO `{$table}` VALUES (" . implode(', ', $values) . ");\n";
                    }
                }
                $output .= "\n";
            }
        }
        
        if ($format === 'mysql') {
            $output .= "/*!40101 SET CHARACTER_SET_CLIENT=@OLD_CHARACTER_SET_CLIENT */;\n";
            $output .= "/*!40101 SET CHARACTER_SET_RESULTS=@OLD_CHARACTER_SET_RESULTS */;\n";
            $output .= "/*!40101 SET COLLATION_CONNECTION=@OLD_COLLATION_CONNECTION */;\n";
        }
        
        file_put_contents($filepath, $output);
        return ['success' => true];
    } catch (Exception $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

// Function: Convert SQLite CREATE TABLE to MySQL format
function convertSQLiteToMySQL($sqliteSQL, $tableName) {
    // Parse SQLite CREATE TABLE statement
    // Extract column definitions
    if (!preg_match('/CREATE TABLE\s+[`"]?' . preg_quote($tableName, '/') . '[`"]?\s*\((.*)\)/is', $sqliteSQL, $matches)) {
        // Fallback: simple replacement
        $mysqlSQL = $sqliteSQL;
        $mysqlSQL = preg_replace('/INTEGER PRIMARY KEY AUTOINCREMENT/i', 'INT AUTO_INCREMENT', $mysqlSQL);
        $mysqlSQL = preg_replace('/INTEGER/i', 'INT', $mysqlSQL);
        $mysqlSQL = preg_replace('/TEXT PRIMARY KEY/i', 'VARCHAR(255) PRIMARY KEY', $mysqlSQL);
        $mysqlSQL = preg_replace('/TEXT/i', 'TEXT', $mysqlSQL);
        $mysqlSQL = preg_replace('/REAL/i', 'DOUBLE', $mysqlSQL);
        $mysqlSQL = preg_replace('/BLOB/i', 'BLOB', $mysqlSQL);
        // Remove CHECK constraints (not supported in MySQL < 8.0.16)
        $mysqlSQL = preg_replace('/CHECK\s*\([^)]+\)/i', '', $mysqlSQL);
        $mysqlSQL = rtrim($mysqlSQL, ';');
        $mysqlSQL .= " ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        return $mysqlSQL;
    }
    
    $columnDefs = trim($matches[1]);
    
    // Split by comma (but not inside parentheses)
    $columns = [];
    $depth = 0;
    $current = '';
    for ($i = 0; $i < strlen($columnDefs); $i++) {
        $char = $columnDefs[$i];
        if ($char === '(') $depth++;
        if ($char === ')') $depth--;
        if ($char === ',' && $depth === 0) {
            $columns[] = trim($current);
            $current = '';
        } else {
            $current .= $char;
        }
    }
    if ($current) $columns[] = trim($current);
    
    // Convert each column
    $mysqlColumns = [];
    $primaryKey = null;
    
    foreach ($columns as $col) {
        $col = trim($col);
        
        // Skip empty
        if (empty($col)) continue;
        
        // Skip CHECK constraints (not supported in MySQL < 8.0.16)
        if (preg_match('/^CHECK\s*\(/i', $col)) {
            continue;
        }
        
        // Check if it's a PRIMARY KEY constraint
        if (preg_match('/^PRIMARY KEY\s*\(/i', $col)) {
            $primaryKey = $col;
            continue;
        }
        
        // Convert INTEGER PRIMARY KEY AUTOINCREMENT
        if (preg_match('/^([`"]?\w+[`"]?)\s+INTEGER\s+PRIMARY\s+KEY\s+AUTOINCREMENT/i', $col, $m)) {
            $mysqlColumns[] = $m[1] . ' INT AUTO_INCREMENT PRIMARY KEY';
            continue;
        }
        
        // Convert TEXT PRIMARY KEY to VARCHAR(255) PRIMARY KEY
        if (preg_match('/^([`"]?\w+[`"]?)\s+TEXT\s+PRIMARY\s+KEY/i', $col, $m)) {
            $mysqlColumns[] = $m[1] . ' VARCHAR(255) PRIMARY KEY';
            continue;
        }
        
        // Convert INT PRIMARY KEY (without AUTOINCREMENT)
        if (preg_match('/^([`"]?\w+[`"]?)\s+INT(?:EGER)?\s+PRIMARY\s+KEY(?!\s+AUTO)/i', $col, $m)) {
            // Remove CHECK constraint if exists
            $col = preg_replace('/CHECK\s*\([^)]+\)/i', '', $col);
            $mysqlColumns[] = trim($col);
            continue;
        }
        
        // Convert data types
        $col = preg_replace('/\bINTEGER\b/i', 'INT', $col);
        $col = preg_replace('/\bREAL\b/i', 'DOUBLE', $col);
        
        // Remove CHECK constraints from column definition
        $col = preg_replace('/CHECK\s*\([^)]+\)/i', '', $col);
        $col = trim($col);
        
        if (!empty($col)) {
            $mysqlColumns[] = $col;
        }
    }
    
    // Add PRIMARY KEY constraint if exists
    if ($primaryKey) {
        $mysqlColumns[] = $primaryKey;
    }
    
    // Add backticks to column names if not already present
    $mysqlColumns = array_map(function($col) {
        // Check if column name already has backticks
        if (preg_match('/^`/', trim($col))) {
            return $col;
        }
        // Add backticks to column name (first word)
        if (preg_match('/^(\w+)(\s+.*)$/', trim($col), $m)) {
            return '`' . $m[1] . '`' . $m[2];
        }
        return $col;
    }, $mysqlColumns);
    
    // Build MySQL CREATE TABLE
    $mysqlSQL = "CREATE TABLE `{$tableName}` (\n  ";
    $mysqlSQL .= implode(",\n  ", $mysqlColumns);
    $mysqlSQL .= "\n) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    return $mysqlSQL;
}

// Get list of backups
$backups = [];
if (is_dir($backupDir)) {
    $files = array_merge(
        glob($backupDir . '/*.db'),
        glob($backupDir . '/*.sql')
    );
    foreach ($files as $file) {
        $ext = pathinfo($file, PATHINFO_EXTENSION);
        $backups[] = [
            'filename' => basename($file),
            'size' => filesize($file),
            'date' => filemtime($file),
            'type' => $ext === 'sql' ? 'SQL' : 'SQLite'
        ];
    }
    // Sort by date descending
    usort($backups, function($a, $b) {
        return $b['date'] - $a['date'];
    });
}

// Get database info
$dbSize = file_exists($dbFile) ? filesize($dbFile) : 0;
$db = get_db();
$tables = $db->query("SELECT name FROM sqlite_master WHERE type='table' ORDER BY name")->fetchAll(PDO::FETCH_COLUMN);
$tableCount = count($tables);

// Helper function: Format file size
function formatFileSize($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    return round($bytes / pow(1024, $pow), 2) . ' ' . $units[$pow];
}

include '_header.php';
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-md-12">
            <h2><i class="fas fa-database"></i> <?php echo $pageTitle; ?></h2>
            <hr>
            
            <?php if ($success): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    <i class="fas fa-check-circle"></i> <?php echo $success; ?>
                    <button type="button" class="close" data-dismiss="alert">&times;</button>
                </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
                <div class="alert alert-danger alert-dismissible fade show">
                    <i class="fas fa-exclamation-circle"></i> <?php echo $error; ?>
                    <button type="button" class="close" data-dismiss="alert">&times;</button>
                </div>
            <?php endif; ?>
            
            <!-- Backup Actions -->
            <div class="row mb-4">
                <div class="col-md-6">
                    <div class="card border-primary">
                        <div class="card-header bg-primary text-white">
                            <h5><i class="fas fa-download"></i> Backup Database</h5>
                        </div>
                        <div class="card-body">
                            <p>Buat backup database saat ini. Pilih format backup:</p>
                            
                            <!-- Backup SQLite Format -->
                            <form method="POST" onsubmit="return confirm('Yakin ingin membuat backup database (format SQLite)?');" class="mb-2">
                                <input type="hidden" name="action" value="backup">
                                <button type="submit" class="btn btn-primary btn-lg btn-block">
                                    <i class="fas fa-database"></i> Backup Format SQLite (.db)
                                </button>
                            </form>
                            
                            <!-- Backup SQL Format -->
                            <form method="POST" id="sqlBackupForm">
                                <input type="hidden" name="action" value="backup_sql">
                                <div class="form-group mb-2">
                                    <label class="font-weight-bold">Format SQL:</label>
                                    <div class="form-check">
                                        <input class="form-check-input" type="radio" name="sql_format" id="sql_sqlite" value="sqlite" checked>
                                        <label class="form-check-label" for="sql_sqlite">
                                            <i class="fas fa-database text-info"></i> SQLite SQL
                                        </label>
                                        <small class="form-text text-muted">Untuk restore di aplikasi ini</small>
                                    </div>
                                    <div class="form-check">
                                        <input class="form-check-input" type="radio" name="sql_format" id="sql_mysql" value="mysql">
                                        <label class="form-check-label" for="sql_mysql">
                                            <i class="fas fa-server text-warning"></i> MySQL/MariaDB SQL
                                        </label>
                                        <small class="form-text text-muted">Untuk phpMyAdmin & MySQL</small>
                                    </div>
                                </div>
                                <button type="submit" class="btn btn-success btn-lg btn-block" onclick="return confirm('Yakin ingin membuat backup database (format SQL)?');">
                                    <i class="fas fa-file-code"></i> Backup Format SQL (.sql)
                                </button>
                            </form>
                            
                            <small class="text-muted mt-3 d-block">
                                <i class="fas fa-info-circle"></i> Database: <strong>school.db</strong> (<?php echo formatFileSize($dbSize); ?>)
                            </small>
                            <div class="alert alert-info mt-2 mb-0">
                                <small>
                                    <strong>Format Backup:</strong><br>
                                    <i class="fas fa-database"></i> <strong>SQLite (.db):</strong> Backup cepat, restore langsung<br>
                                    <i class="fas fa-file-code"></i> <strong>SQLite SQL:</strong> Portable, bisa diedit<br>
                                    <i class="fas fa-server"></i> <strong>MySQL SQL:</strong> Upload ke phpMyAdmin ✓
                                </small>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-md-6">
                    <div class="card border-info">
                        <div class="card-header bg-info text-white">
                            <h5><i class="fas fa-info-circle"></i> Informasi Database</h5>
                        </div>
                        <div class="card-body">
                            <ul class="list-unstyled">
                                <li><i class="fas fa-database text-primary"></i> Ukuran Database: <strong><?php echo formatFileSize($dbSize); ?></strong></li>
                                <li><i class="fas fa-table text-primary"></i> Jumlah Tabel: <strong><?php echo $tableCount; ?></strong></li>
                                <li><i class="fas fa-save text-success"></i> Total Backup: <strong><?php echo count($backups); ?></strong></li>
                                <li><i class="fas fa-folder text-warning"></i> Lokasi: <code>backups/database/</code></li>
                            </ul>
                            <div class="alert alert-warning mb-0">
                                <small><i class="fas fa-exclamation-triangle"></i> <strong>Penting:</strong> Selalu backup sebelum melakukan perubahan besar pada database!</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Backup List -->
            <div class="card">
                <div class="card-header bg-dark text-white">
                    <h5><i class="fas fa-list"></i> Daftar Backup</h5>
                </div>
                <div class="card-body">
                    <?php if (empty($backups)): ?>
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle"></i> Belum ada backup. Silakan buat backup terlebih dahulu.
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead class="thead-dark">
                                    <tr>
                                        <th width="5%">#</th>
                                        <th width="30%">Nama File</th>
                                        <th width="10%">Tipe</th>
                                        <th width="12%">Ukuran</th>
                                        <th width="18%">Tanggal</th>
                                        <th width="25%" class="text-center">Aksi</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($backups as $index => $backup): ?>
                                        <tr>
                                            <td><?php echo $index + 1; ?></td>
                                            <td>
                                                <i class="fas <?php echo $backup['type'] === 'SQL' ? 'fa-file-code text-success' : 'fa-database text-primary'; ?>"></i>
                                                <code><?php echo htmlspecialchars($backup['filename']); ?></code>
                                            </td>
                                            <td>
                                                <span class="badge badge-<?php echo $backup['type'] === 'SQL' ? 'success' : 'primary'; ?>">
                                                    <?php echo $backup['type']; ?>
                                                </span>
                                            </td>
                                            <td><?php echo formatFileSize($backup['size']); ?></td>
                                            <td><?php echo date('d-m-Y H:i:s', $backup['date']); ?></td>
                                            <td class="text-center">
                                                <!-- Download -->
                                                <form method="POST" style="display:inline;">
                                                    <input type="hidden" name="action" value="download">
                                                    <input type="hidden" name="filename" value="<?php echo htmlspecialchars($backup['filename']); ?>">
                                                    <button type="submit" class="btn btn-sm btn-success" title="Download">
                                                        <i class="fas fa-download"></i>
                                                    </button>
                                                </form>
                                                
                                                <!-- Restore -->
                                                <form method="POST" style="display:inline;" onsubmit="return confirm('PERINGATAN: Restore akan menimpa database saat ini!\n\nYakin ingin melanjutkan?');">
                                                    <input type="hidden" name="action" value="restore">
                                                    <input type="hidden" name="filename" value="<?php echo htmlspecialchars($backup['filename']); ?>">
                                                    <button type="submit" class="btn btn-sm btn-warning" title="Restore">
                                                        <i class="fas fa-upload"></i>
                                                    </button>
                                                </form>
                                                
                                                <!-- Delete -->
                                                <form method="POST" style="display:inline;" onsubmit="return confirm('Yakin ingin menghapus backup ini?');">
                                                    <input type="hidden" name="action" value="delete">
                                                    <input type="hidden" name="filename" value="<?php echo htmlspecialchars($backup['filename']); ?>">
                                                    <button type="submit" class="btn btn-sm btn-danger" title="Hapus">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </form>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Help Section -->
            <div class="card mt-4 border-warning">
                <div class="card-header bg-warning">
                    <h5><i class="fas fa-question-circle"></i> Panduan Penggunaan</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <h6><i class="fas fa-download text-primary"></i> Backup</h6>
                            <ol>
                                <li>Pilih format backup (SQLite atau SQL)</li>
                                <li>Klik tombol backup yang diinginkan</li>
                                <li>File backup akan dibuat otomatis</li>
                                <li>Backup tersimpan di folder <code>backups/database/</code></li>
                            </ol>
                            <p><strong>Format Backup:</strong></p>
                            <ul>
                                <li><strong>.db</strong> - Binary SQLite (cepat, restore langsung)</li>
                                <li><strong>.sql (SQLite)</strong> - Text SQL untuk SQLite</li>
                                <li><strong>.sql (MySQL)</strong> - Compatible dengan phpMyAdmin ✓</li>
                            </ul>
                            <div class="alert alert-success">
                                <small><i class="fas fa-check-circle"></i> <strong>Untuk phpMyAdmin:</strong> Pilih format "MySQL/MariaDB SQL" saat backup</small>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <h6><i class="fas fa-upload text-warning"></i> Restore</h6>
                            <ol>
                                <li>Pilih backup yang ingin di-restore</li>
                                <li>Klik tombol <span class="badge badge-warning"><i class="fas fa-upload"></i></span></li>
                                <li>Konfirmasi restore</li>
                                <li>Database akan dikembalikan ke kondisi backup</li>
                            </ol>
                        </div>
                    </div>
                    <hr>
                    <div class="alert alert-danger mb-0">
                        <strong><i class="fas fa-exclamation-triangle"></i> PERINGATAN:</strong>
                        <ul class="mb-0">
                            <li>Restore akan menimpa semua data database saat ini</li>
                            <li>Sistem otomatis membuat backup sementara sebelum restore</li>
                            <li>Pastikan membuat backup terlebih dahulu sebelum restore</li>
                            <li>Download backup penting untuk disimpan di tempat aman</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.card {
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    margin-bottom: 20px;
}
.btn-lg {
    padding: 15px;
    font-size: 18px;
}
code {
    background: #f4f4f4;
    padding: 2px 6px;
    border-radius: 3px;
}
</style>

<?php include '_footer.php'; ?>
